/*
 * Decompiled with CFR 0.152.
 */
package greymerk.roguelike.dungeon;

import com.google.common.base.Charsets;
import com.google.common.collect.Lists;
import com.google.common.io.Files;
import greymerk.roguelike.config.RogueConfig;
import greymerk.roguelike.dungeon.DungeonLevel;
import greymerk.roguelike.dungeon.DungeonStage;
import greymerk.roguelike.dungeon.settings.DungeonSettings;
import greymerk.roguelike.dungeon.settings.SettingIdentifier;
import greymerk.roguelike.dungeon.settings.SettingsContainer;
import greymerk.roguelike.dungeon.settings.SettingsRandom;
import greymerk.roguelike.dungeon.settings.SettingsResolver;
import greymerk.roguelike.dungeon.settings.SpawnCriteria;
import greymerk.roguelike.dungeon.tasks.DungeonTaskRegistry;
import greymerk.roguelike.treasure.ITreasureChest;
import greymerk.roguelike.worldgen.Cardinal;
import greymerk.roguelike.worldgen.Coord;
import greymerk.roguelike.worldgen.IWorldEditor;
import greymerk.roguelike.worldgen.VanillaStructure;
import greymerk.roguelike.worldgen.shapes.RectSolid;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Random;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import net.minecraft.block.material.Material;
import net.minecraft.client.Minecraft;
import net.minecraft.world.biome.Biome;
import net.minecraftforge.common.BiomeDictionary;
import org.apache.commons.io.FilenameUtils;

public class Dungeon {
    public static final int VERTICAL_SPACING = 10;
    public static final int TOPLEVEL = 50;
    private static final String SETTINGS_DIRECTORY = "config/roguelike_dungeons/settings";
    public static SettingsResolver settingsResolver;
    private Coord origin;
    private List<DungeonLevel> levels = new ArrayList<DungeonLevel>();
    private IWorldEditor editor;

    public Dungeon(IWorldEditor editor) {
        this.editor = editor;
    }

    public static void initResolver() throws Exception {
        File settingsDirectory = new File(SETTINGS_DIRECTORY);
        if (settingsDirectory.exists() && !settingsDirectory.isDirectory()) {
            throw new Exception("Settings directory is a file");
        }
        if (!settingsDirectory.exists()) {
            settingsDirectory.mkdir();
        }
        Map<String, String> fileByName = Dungeon.collectSettingsFiles(settingsDirectory);
        SettingsContainer settings = new SettingsContainer(new DungeonSettings[0]);
        settings.put(fileByName);
        settingsResolver = new SettingsResolver(settings);
    }

    private static Map<String, String> collectSettingsFiles(File settingsDirectory) {
        List<File> files = Dungeon.listFilesRecursively(settingsDirectory);
        return Dungeon.mapContentByFilename(files);
    }

    private static List<File> listFilesRecursively(File settingsDirectory) {
        Object[] files = settingsDirectory.listFiles();
        return Optional.ofNullable(files).isPresent() ? Lists.newArrayList((Object[])files).stream().flatMap(file -> file.isDirectory() ? Dungeon.listFilesRecursively(file).stream() : Lists.newArrayList((Object[])new File[]{file}).stream()).filter(file -> FilenameUtils.getExtension((String)file.getName()).equals("json")).collect(Collectors.toList()) : Collections.emptyList();
    }

    private static Map<String, String> mapContentByFilename(List<File> files) {
        return files.stream().collect(Collectors.toMap(File::getName, Dungeon::getFileContent));
    }

    private static String getFileContent(File file) {
        try {
            return Files.toString((File)file, (Charset)Charsets.UTF_8);
        }
        catch (IOException e) {
            throw new RuntimeException("Error reading file : " + file.getName());
        }
    }

    public static boolean canSpawnInChunk(int chunkX, int chunkZ, IWorldEditor editor) {
        return RogueConfig.getBoolean(RogueConfig.DONATURALSPAWN) && SpawnCriteria.isValidDimension(Dungeon.getDimension(chunkX, chunkZ, editor)) && Dungeon.isVillageChunk(editor, chunkX, chunkZ) && Dungeon.isSpawnChanceHit(chunkX, chunkZ);
    }

    private static int getDimension(int chunkX, int chunkZ, IWorldEditor editor) {
        Coord coord = new Coord(chunkX * 16, 0, chunkZ * 16);
        return editor.getInfo(coord).getDimension();
    }

    public static boolean isVillageChunk(IWorldEditor editor, int chunkX, int chunkZ) {
        int frequency = RogueConfig.getInt(RogueConfig.SPAWNFREQUENCY);
        int min = 8 * frequency / 10;
        int max = 32 * frequency / 10;
        min = Math.max(min, 2);
        max = Math.max(max, 8);
        int tempX = chunkX < 0 ? chunkX - (max - 1) : chunkX;
        int tempZ = chunkZ < 0 ? chunkZ - (max - 1) : chunkZ;
        int m = tempX / max;
        int n = tempZ / max;
        Random r = editor.getSeededRandom(m, n, 10387312);
        m *= max;
        n *= max;
        return chunkX == (m += r.nextInt(max - min)) && chunkZ == (n += r.nextInt(max - min));
    }

    private static boolean isSpawnChanceHit(int chunkX, int chunkZ) {
        double spawnChance = RogueConfig.getDouble(RogueConfig.SPAWNCHANCE);
        Random rand = new Random(Objects.hash(chunkX, chunkZ, 31));
        return (double)rand.nextFloat() < spawnChance;
    }

    public static int getLevel(int y) {
        if (y < 15) {
            return 4;
        }
        if (y < 25) {
            return 3;
        }
        if (y < 35) {
            return 2;
        }
        if (y < 45) {
            return 1;
        }
        return 0;
    }

    public static Coord getNearbyCoord(Random random, int x, int z, int min, int max) {
        int distance = min + random.nextInt(max - min);
        double angle = random.nextDouble() * 2.0 * Math.PI;
        int xOffset = (int)(Math.cos(angle) * (double)distance);
        int zOffset = (int)(Math.sin(angle) * (double)distance);
        return new Coord(x + xOffset, 0, z + zOffset);
    }

    public static Random getRandom(IWorldEditor editor, Coord pos) {
        return new Random(Objects.hash(editor.getSeed(), pos));
    }

    private Optional<Coord> selectLocation(Random rand, int x, int z) {
        return IntStream.range(0, 50).mapToObj(i -> Dungeon.getNearbyCoord(rand, x, z, 40, 100)).filter(this::canGenerateDungeonHere).findFirst();
    }

    private void printDungeonName(DungeonSettings dungeonSettings) {
        Optional<SettingIdentifier> id = Optional.ofNullable(dungeonSettings.getId());
        String string = id.isPresent() ? id.toString() : dungeonSettings.toString();
        Minecraft.func_71410_x().field_71439_g.func_71165_d(string);
    }

    public void generate(DungeonSettings dungeonSettings, Coord coord) {
        try {
            Random random = Dungeon.getRandom(this.editor, coord);
            this.origin = new Coord(coord.getX(), 50, coord.getZ());
            Coord start = this.getPosition();
            IntStream.range(0, dungeonSettings.getNumLevels()).mapToObj(dungeonSettings::getLevelSettings).map(DungeonLevel::new).forEach(this.levels::add);
            Arrays.stream(DungeonStage.values()).flatMap(stage -> DungeonTaskRegistry.getTaskRegistry().getTasks((DungeonStage)((Object)stage)).stream()).forEach(task -> task.execute(this.editor, random, this, dungeonSettings));
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void spawnInChunk(Random rand, int chunkX, int chunkZ) {
        if (Dungeon.canSpawnInChunk(chunkX, chunkZ, this.editor)) {
            int x = chunkX * 16 + 4;
            int z = chunkZ * 16 + 4;
            this.selectLocation(rand, x, z).ifPresent(coord -> this.getDungeonSettingsMaybe((Coord)coord).ifPresent(dungeonSettings -> this.generate((DungeonSettings)dungeonSettings, (Coord)coord)));
        }
    }

    private Optional<DungeonSettings> getDungeonSettingsMaybe(Coord coord) {
        if (RogueConfig.getBoolean(RogueConfig.RANDOM)) {
            return Optional.of(new SettingsRandom(Dungeon.getRandom(this.editor, coord)));
        }
        if (settingsResolver != null) {
            return Optional.ofNullable(settingsResolver.getAnyCustomDungeonSettings(this.editor, coord));
        }
        return Optional.empty();
    }

    public boolean canGenerateDungeonHere(Coord coord) {
        if (this.isInvalidBiome(coord) || this.hasStrongholdTooCloseBy(coord)) {
            return false;
        }
        int upperLimit = RogueConfig.getInt(RogueConfig.UPPERLIMIT);
        int lowerLimit = RogueConfig.getInt(RogueConfig.LOWERLIMIT);
        Coord cursor = new Coord(coord.getX(), upperLimit, coord.getZ());
        return this.editor.isAirBlock(cursor) && this.canFindStartingCoord(lowerLimit, cursor) && this.isFreeOverhead(cursor) && this.isSolidBelow(cursor);
    }

    private boolean isInvalidBiome(Coord coord) {
        Biome biome = this.editor.getInfo(coord).getBiome();
        BiomeDictionary.Type[] invalidBiomes = new BiomeDictionary.Type[]{BiomeDictionary.Type.RIVER, BiomeDictionary.Type.BEACH, BiomeDictionary.Type.MUSHROOM, BiomeDictionary.Type.OCEAN};
        return Arrays.stream(invalidBiomes).anyMatch(type -> BiomeDictionary.hasType((Biome)biome, (BiomeDictionary.Type)type));
    }

    private boolean hasStrongholdTooCloseBy(Coord coord) {
        Coord stronghold = this.editor.findNearestStructure(VanillaStructure.STRONGHOLD, coord);
        if (stronghold == null) {
            return false;
        }
        double strongholdDistance = coord.distance(stronghold);
        return strongholdDistance < 300.0;
    }

    private boolean canFindStartingCoord(int lowerLimit, Coord cursor) {
        while (!this.editor.validGroundBlock(cursor)) {
            cursor.translate(Cardinal.DOWN);
            if (cursor.getY() < lowerLimit) {
                return false;
            }
            if (this.editor.getBlock(cursor).func_185904_a() != Material.field_151586_h) continue;
            return false;
        }
        return true;
    }

    private boolean isFreeOverhead(Coord cursor) {
        Coord start = new Coord(cursor).translate(new Coord(-4, 4, -4));
        Coord end = new Coord(cursor).translate(new Coord(4, 4, 4));
        for (Coord c : new RectSolid(start, end)) {
            if (!this.editor.validGroundBlock(c)) continue;
            return false;
        }
        return true;
    }

    private boolean isSolidBelow(Coord cursor) {
        Coord start1 = new Coord(cursor).translate(new Coord(-4, -3, -4));
        Coord end1 = new Coord(cursor).translate(new Coord(4, -3, 4));
        int airCount = 0;
        for (Coord c : new RectSolid(start1, end1)) {
            if (!this.editor.validGroundBlock(c)) {
                ++airCount;
            }
            if (airCount <= 8) continue;
            return false;
        }
        return true;
    }

    public List<ITreasureChest> getChests() {
        return this.editor.getTreasure().getChests();
    }

    public Coord getPosition() {
        return new Coord(this.origin);
    }

    public List<DungeonLevel> getLevels() {
        return this.levels;
    }

    static {
        try {
            RogueConfig.reload(false);
            Dungeon.initResolver();
        }
        catch (Exception exception) {
            // empty catch block
        }
    }
}

